#!/usr/bin/env python3
"""
Classical Nyx vs QAOA Comparison
================================
QIP 2026 Rump Session - Verification Script

This script compares Classical Nyx against QAOA results from IBM Quantum hardware.

QAOA baselines from January 22, 2026 testing:
- 3 IBM backends (Torino 133q, Fez 156q, Marrakesh 156q)
- 5,000 shots per circuit
- Standard QAOA parameters (gamma=0.5, beta=0.5)
- 24 Job IDs provided for independent verification

To verify QAOA results:
1. Go to quantum.ibm.com
2. Navigate to Jobs
3. Enter any Job ID from the output
4. View measurement results

Usage:
    python classical_nyx_vs_qaoa.py

Author: Subvurs Research
Date: January 2026
"""

import sys
try:
    import nyx_core
except ImportError:
    print("ERROR: nyx_core module not found.")
    print("Please build it first:")
    print("    python setup.py build_ext --inplace")
    sys.exit(1)

import numpy as np
from typing import List, Tuple

# =============================================================================
# PROBLEM DEFINITIONS
# =============================================================================

def max_cut_score(state: List[int]) -> int:
    """
    Max-Cut: Partition graph to maximize cut weight.
    4-node graph with weighted edges.
    Optimal: 19
    """
    edges = [(0, 1, 5), (0, 2, 3), (1, 3, 7), (2, 3, 4)]
    return sum(w for i, j, w in edges if state[i] != state[j])


def portfolio_score(state: List[int]) -> float:
    """
    Portfolio Optimization: Maximize return minus risk.
    4 assets with returns and risk penalty.
    Optimal: 0.15
    """
    returns = [0.05, 0.08, 0.12, 0.15]
    risk = 0.5
    ret = sum(returns[i] for i in range(len(state)) if state[i] == 1)
    pairs = sum(1 for i in range(len(state)) for j in range(i+1, len(state))
                if state[i] == 1 and state[j] == 1)
    return ret - risk * pairs


def set_cover_score(state: List[int]) -> Tuple[int, bool]:
    """
    Set Cover: Cover all elements with minimum sets.
    Universe {0,1,2,3}, 4 sets available.
    Optimal: 2 sets
    """
    sets = [[0, 1], [1, 2], [2, 3], [0, 3]]
    universe = {0, 1, 2, 3}
    covered = set()
    for i in range(len(state)):
        if state[i] == 1:
            covered.update(sets[i])
    return sum(state), covered == universe


def knapsack_score(state: List[int]) -> Tuple[int, bool]:
    """
    Knapsack: Maximize value under capacity.
    4 items, capacity 9.
    Optimal: 25
    """
    weights = [2, 3, 4, 5]
    values = [5, 8, 12, 15]
    capacity = 9
    total_weight = sum(weights[i] for i in range(len(state)) if state[i] == 1)
    total_value = sum(values[i] for i in range(len(state)) if state[i] == 1)
    return total_value, total_weight <= capacity


# =============================================================================
# QAOA BASELINES (IBM Quantum - January 22, 2026)
# =============================================================================

QAOA_RESULTS = {
    'max_cut': {
        'ibm_torino': {'optimal_pct': 10.92, 'job_id': 'd5pab0gr0v5s739nmtog'},
        'ibm_fez': {'optimal_pct': 11.72, 'job_id': 'd5pacngr0v5s739nmvlg'},
        'ibm_marrakesh': {'optimal_pct': 9.54, 'job_id': 'd5pad50h0i0s73ep08o0'},
    },
    'portfolio': {
        'ibm_torino': {'optimal_pct': 2.58, 'job_id': 'd5pac1oh0i0s73ep07d0'},
        'ibm_fez': {'optimal_pct': 3.32, 'job_id': 'd5pacqu5v3os73f1od00'},
        'ibm_marrakesh': {'optimal_pct': 3.06, 'job_id': 'd5paj3gh0i0s73ep0fr0'},
    },
    'set_cover': {
        'ibm_torino': {'optimal_pct': 22.92, 'job_id': 'd5pacgu5v3os73f1och0'},
        'ibm_fez': {'optimal_pct': 24.38, 'job_id': 'd5pacu0r0v5s739nn00g'},
        'ibm_marrakesh': {'optimal_pct': 21.52, 'job_id': 'd5pan3pdgvjs73dbg30g'},
    },
    'knapsack': {
        'ibm_torino': {'optimal_pct': 4.48, 'job_id': 'd5pack65v3os73f1oclg'},
        'ibm_fez': {'optimal_pct': 4.28, 'job_id': 'd5pad1e5v3os73f1od8g'},
        'ibm_marrakesh': {'optimal_pct': 4.18, 'job_id': 'd5papo65v3os73f1ot4g'},
    }
}

PROBLEMS = {
    'max_cut': {
        'name': 'Max-Cut',
        'scorer': max_cut_score,
        'optimal': 19,
        'mode': 'mode_a',
    },
    'portfolio': {
        'name': 'Portfolio',
        'scorer': portfolio_score,
        'optimal': 0.15,
        'mode': 'mode_b',
    },
    'set_cover': {
        'name': 'Set Cover',
        'scorer': set_cover_score,
        'optimal': 2,
        'mode': 'mode_c',
    },
    'knapsack': {
        'name': 'Knapsack',
        'scorer': knapsack_score,
        'optimal': 25,
        'mode': 'mode_d',
    }
}


# =============================================================================
# MAIN TEST
# =============================================================================

def run_comparison():
    """Run Classical Nyx vs QAOA comparison."""

    print("=" * 78)
    print("CLASSICAL NYX vs QUANTUM QAOA COMPARISON")
    print("=" * 78)
    print(f"\nNyx Core Version: {nyx_core.get_version()}")
    print(f"Available Modes: {nyx_core.get_available_modes()}")
    print("\nQAOA Baselines: IBM Quantum (January 22, 2026)")
    print("  - Backends: ibm_torino (133q), ibm_fez (156q), ibm_marrakesh (156q)")
    print("  - Shots: 5,000 per circuit")
    print("  - Parameters: gamma=0.5, beta=0.5 (standard)")
    print()

    results = []

    for prob_key, prob in PROBLEMS.items():
        print(f"\n{'─' * 78}")
        print(f"Testing: {prob['name']}")
        print(f"{'─' * 78}")

        qaoa_data = QAOA_RESULTS[prob_key]
        qaoa_values = [v['optimal_pct'] for v in qaoa_data.values()]
        qaoa_avg = np.mean(qaoa_values)

        print(f"\nQAOA Results (from IBM Quantum):")
        for backend, data in qaoa_data.items():
            print(f"  {backend}: {data['optimal_pct']:.2f}%  [Job ID: {data['job_id']}]")
        print(f"  Average: {qaoa_avg:.2f}%")

        print(f"\nRunning Classical Nyx...")

        classical_freq = nyx_core.optimize(
            scorer=prob['scorer'],
            optimal_value=prob['optimal'],
            mode=prob['mode'],
            n_bits=4,
            iterations=20,
            samples=5000
        ) * 100

        print(f"  Classical Nyx: {classical_freq:.2f}%")

        if qaoa_avg > 0:
            improvement = ((classical_freq - qaoa_avg) / qaoa_avg) * 100
            ratio = classical_freq / qaoa_avg
        else:
            improvement = float('inf')
            ratio = float('inf')

        winner = "CLASSICAL" if classical_freq > qaoa_avg else "QAOA" if qaoa_avg > classical_freq else "TIE"

        print(f"\n  Result: {winner}")
        print(f"  Improvement: {improvement:+.1f}%")
        print(f"  Ratio: {ratio:.2f}x")

        results.append({
            'problem': prob['name'],
            'classical': classical_freq,
            'qaoa_avg': qaoa_avg,
            'improvement': improvement,
            'ratio': ratio,
            'winner': winner
        })

    print("\n" + "=" * 78)
    print("SUMMARY")
    print("=" * 78)
    print(f"\n{'Problem':<12} {'Classical':<12} {'QAOA Avg':<12} {'Improvement':<14} {'Winner':<10}")
    print("─" * 60)

    classical_wins = 0
    for r in results:
        print(f"{r['problem']:<12} {r['classical']:<12.2f} {r['qaoa_avg']:<12.2f} {r['improvement']:>+13.1f}% {r['winner']:<10}")
        if r['winner'] == "CLASSICAL":
            classical_wins += 1

    print("─" * 60)
    print(f"\nClassical Nyx wins: {classical_wins}/4 problems")

    print("\n" + "=" * 78)
    print("VERIFICATION")
    print("=" * 78)
    print("""
All QAOA results can be independently verified on IBM Quantum Platform:

1. Go to: https://quantum.ibm.com
2. Sign in with your IBM Quantum account
3. Navigate to: Workloads > Jobs
4. Enter any Job ID from above
5. View measurement results

The Classical Nyx results were generated on your local machine.
Run this script multiple times to verify consistency.
""")


if __name__ == '__main__':
    run_comparison()
