"""
Unit tests for hive_keyboard.core module.
"""

import pytest
from hive_keyboard import HiveKeyboard, HIVE_LAYERS, CRITICAL_PATTERNS


class TestHiveKeyboard:
    """Tests for the HiveKeyboard class."""

    def test_initialization_default(self):
        """Test default initialization."""
        hive = HiveKeyboard()
        assert hive.backend_name == 'simulator'
        assert hive.double_twist is True
        assert hive.auto_mitigation is False

    def test_initialization_custom(self):
        """Test custom initialization."""
        hive = HiveKeyboard(backend='ibm_torino', double_twist=False, auto_mitigation=True)
        assert hive.backend_name == 'ibm_torino'
        assert hive.double_twist is False
        assert hive.auto_mitigation is True

    def test_inverse_map_loaded(self):
        """Test that inverse map is loaded with 256 entries."""
        hive = HiveKeyboard()
        assert len(hive.inverse_map) == 256
        assert all(0 <= v <= 255 for v in hive.inverse_map.values())

    def test_inverse_map_bijective(self):
        """Test that inverse map is a bijection (one-to-one)."""
        hive = HiveKeyboard()
        values = list(hive.inverse_map.values())
        assert len(set(values)) == 256, "Inverse map should be bijective"

    def test_inverse_method(self):
        """Test the inverse() method."""
        hive = HiveKeyboard()
        # Known values from testing
        assert hive.inverse(126) == 65
        assert hive.inverse(51) == 170
        assert hive.inverse(0) == 0
        # Note: inverse(255) = 64, not 255 (permutation doesn't preserve 255)

    def test_target_valid_pattern(self):
        """Test targeting valid patterns."""
        hive = HiveKeyboard()
        result = hive.target(126, shots=100)

        assert result['target'] == 126
        assert result['fidelity'] == 100.0
        assert result['hits'] == 100
        assert result['shots'] == 100
        assert result['status'] == 'COMPLETED'

    def test_target_all_256_patterns(self):
        """Test that all 256 patterns achieve 100% fidelity on simulator."""
        hive = HiveKeyboard()

        for pattern_id in range(256):
            result = hive.target(pattern_id, shots=100)
            assert result['fidelity'] == 100.0, f"Pattern {pattern_id} failed"

    def test_target_invalid_pattern_negative(self):
        """Test that negative patterns raise ValueError."""
        hive = HiveKeyboard()
        with pytest.raises(ValueError):
            hive.target(-1)

    def test_target_invalid_pattern_too_large(self):
        """Test that patterns > 255 raise ValueError."""
        hive = HiveKeyboard()
        with pytest.raises(ValueError):
            hive.target(256)

    def test_sequence(self):
        """Test running a sequence of patterns."""
        hive = HiveKeyboard()
        results = hive.sequence([51, 126, 155], shots=100)

        assert len(results) == 3
        assert all(r['fidelity'] == 100.0 for r in results)

    def test_creation_sequence(self):
        """Test the Creation Sequence (P51 → P126 → P155 → P247)."""
        hive = HiveKeyboard()
        results = hive.creation_sequence(shots=100)

        assert len(results) == 4
        assert results[0]['target'] == 51
        assert results[1]['target'] == 126
        assert results[2]['target'] == 155
        assert results[3]['target'] == 247
        assert all(r['fidelity'] == 100.0 for r in results)

    def test_get_circuit(self):
        """Test getting a circuit for a pattern."""
        hive = HiveKeyboard()
        circuit = hive.get_circuit(126)

        assert circuit.num_qubits == 8
        assert circuit.num_clbits == 8

    def test_layer_simple(self):
        """Test layer() method with simple output."""
        hive = HiveKeyboard()

        # Test various layers (names obfuscated for IP protection)
        # L0: 0, L1: 1-50, L2: 51-99, L3: 100-125, L4: 126-130, L5: 131-198, L6: 199-254, L7: 255
        assert 'Layer 0' in hive.layer(0)
        assert 'Layer 2' in hive.layer(51)
        assert 'Layer 4' in hive.layer(126)
        assert 'Layer 7' in hive.layer(255)

    def test_layer_detailed(self):
        """Test layer() method with detailed output."""
        hive = HiveKeyboard()
        info = hive.layer(126, detailed=True)

        assert info['layer'] == 4
        assert info['name'] == 'L4'  # IP-protected naming
        assert info['pattern'] == 126
        assert info['is_critical'] is True
        assert info['critical_name'] == 'P126'  # IP-protected naming

    def test_info(self):
        """Test info() method."""
        hive = HiveKeyboard()
        info = hive.info(51)

        assert info['pattern'] == 51
        assert info['binary'] == '00110011'
        assert info['init_pattern'] == 170
        assert info['is_critical'] is True
        assert 'circuit' in info
        assert 'layer' in info

    def test_critical_patterns_defined(self):
        """Test critical patterns are defined."""
        hive = HiveKeyboard()
        critical = hive.critical_patterns

        # Critical patterns should be non-empty
        assert len(critical) > 0
        assert all(0 <= p <= 255 for p in critical.keys())

    def test_critical_patterns_property(self):
        """Test critical_patterns property."""
        hive = HiveKeyboard()
        critical = hive.critical_patterns

        # IP-protected: critical patterns exist but names are obfuscated
        assert len(critical) > 0
        # Values are now in format 'Pxx'
        for key, val in critical.items():
            assert val.startswith('P')

    def test_repr(self):
        """Test string representation."""
        hive = HiveKeyboard()
        assert "HiveKeyboard" in repr(hive)
        assert "simulator" in repr(hive)


class TestHiveLayers:
    """Tests for HIVE_LAYERS constant."""

    def test_layer_count(self):
        """Test that there are 8 layers (0-7)."""
        assert len(HIVE_LAYERS) == 8

    def test_layer_coverage(self):
        """Test that layers cover all 256 patterns."""
        covered = set()
        for info in HIVE_LAYERS.values():
            low, high = info['range']
            covered.update(range(low, high + 1))

        assert covered == set(range(256))

    def test_layer_structure(self):
        """Test that each layer has required fields."""
        for layer_num, info in HIVE_LAYERS.items():
            assert 'name' in info
            assert 'range' in info
            assert 'description' in info


class TestCriticalPatterns:
    """Tests for CRITICAL_PATTERNS constant."""

    def test_key_patterns_present(self):
        """Test that key patterns are defined."""
        assert 0 in CRITICAL_PATTERNS  # Singularity
        assert 51 in CRITICAL_PATTERNS  # State51 Bridge
        assert 126 in CRITICAL_PATTERNS  # Core
        assert 255 in CRITICAL_PATTERNS  # Horizon

    def test_all_in_valid_range(self):
        """Test that all critical patterns are in valid range."""
        assert all(0 <= p <= 255 for p in CRITICAL_PATTERNS.keys())
