"""
Unit tests for hive_keyboard.circuits module.
"""

import pytest
import numpy as np
from hive_keyboard import build_deterministic_circuit, build_sequence_circuit, get_circuit_info


class TestBuildDeterministicCircuit:
    """Tests for build_deterministic_circuit function."""

    def test_basic_circuit_creation(self):
        """Test basic circuit creation."""
        qc = build_deterministic_circuit(65)

        assert qc.num_qubits == 8
        assert qc.num_clbits == 8

    def test_circuit_with_double_twist(self):
        """Test circuit with double twist enabled."""
        qc = build_deterministic_circuit(65, double_twist=True)
        info = get_circuit_info(qc)

        # Should have 2 CP gates
        assert info['gates'].get('cp', 0) == 2

    def test_circuit_without_double_twist(self):
        """Test circuit with double twist disabled."""
        qc = build_deterministic_circuit(65, double_twist=False)
        info = get_circuit_info(qc)

        # Should have 1 CP gate
        assert info['gates'].get('cp', 0) == 1

    def test_circuit_depth_difference(self):
        """Test that double twist increases depth by 1."""
        qc_single = build_deterministic_circuit(65, double_twist=False)
        qc_double = build_deterministic_circuit(65, double_twist=True)

        assert qc_double.depth() == qc_single.depth() + 1

    def test_circuit_without_measurement(self):
        """Test circuit creation without measurement gates."""
        qc = build_deterministic_circuit(65, include_measurement=False)
        info = get_circuit_info(qc)

        assert info['gates'].get('measure', 0) == 0
        assert qc.num_clbits == 8  # Classical register still created

    def test_circuit_with_measurement(self):
        """Test circuit creation with measurement gates."""
        qc = build_deterministic_circuit(65, include_measurement=True)
        info = get_circuit_info(qc)

        assert info['gates'].get('measure', 0) == 8

    def test_circuit_components(self):
        """Test that circuit has expected gate types."""
        qc = build_deterministic_circuit(65, double_twist=True)
        info = get_circuit_info(qc)

        # Expected gates: X (init), CCX (Toffoli), RZ (phase), CX (Möbius), CP (twist), measure
        assert 'ccx' in info['gates']  # Toffoli
        assert 'rz' in info['gates']   # Phase lock
        assert 'cx' in info['gates']   # Möbius fold CX chain
        assert 'cp' in info['gates']   # Pi twist

    def test_different_init_patterns(self):
        """Test circuit creation with different init patterns."""
        for init in [0, 65, 127, 170, 255]:
            qc = build_deterministic_circuit(init)
            assert qc.num_qubits == 8

    def test_custom_qubit_count(self):
        """Test circuit creation with custom qubit count."""
        qc = build_deterministic_circuit(15, n_qubits=4)

        assert qc.num_qubits == 4
        assert qc.num_clbits == 4


class TestBuildSequenceCircuit:
    """Tests for build_sequence_circuit function."""

    def test_sequence_circuit_creation(self):
        """Test creating a sequence of circuits."""
        inits = [170, 65, 90, 88]  # Inits for Creation Sequence
        circuits = build_sequence_circuit(inits)

        assert len(circuits) == 4
        assert all(qc.num_qubits == 8 for qc in circuits)

    def test_sequence_empty(self):
        """Test creating empty sequence."""
        circuits = build_sequence_circuit([])
        assert len(circuits) == 0

    def test_sequence_single(self):
        """Test creating single-element sequence."""
        circuits = build_sequence_circuit([65])
        assert len(circuits) == 1


class TestGetCircuitInfo:
    """Tests for get_circuit_info function."""

    def test_info_structure(self):
        """Test that info dict has expected keys."""
        qc = build_deterministic_circuit(65)
        info = get_circuit_info(qc)

        assert 'depth' in info
        assert 'gate_count' in info
        assert 'gates' in info
        assert 'num_qubits' in info
        assert 'num_clbits' in info

    def test_info_values(self):
        """Test info values are reasonable."""
        qc = build_deterministic_circuit(65, double_twist=True)
        info = get_circuit_info(qc)

        assert info['num_qubits'] == 8
        assert info['num_clbits'] == 8
        assert info['depth'] > 0
        assert info['gate_count'] > 0
        assert isinstance(info['gates'], dict)

    def test_gate_count_consistency(self):
        """Test that gate_count equals sum of individual gates."""
        qc = build_deterministic_circuit(65)
        info = get_circuit_info(qc)

        assert info['gate_count'] == sum(info['gates'].values())
