"""
Unit tests for hive_keyboard.backends module.
"""

import pytest
from hive_keyboard import run_simulator, run_batch, HiveKeyboard


class TestRunSimulator:
    """Tests for run_simulator function."""

    def test_basic_simulation(self):
        """Test basic simulator run."""
        hive = HiveKeyboard()
        result = run_simulator(126, hive.inverse_map, shots=100)

        assert result['target'] == 126
        assert result['fidelity'] == 100.0
        assert result['hits'] == 100
        assert result['shots'] == 100
        assert result['status'] == 'COMPLETED'
        assert result['backend'] == 'aer_simulator'

    def test_simulation_binary_format(self):
        """Test that binary format is correct."""
        hive = HiveKeyboard()
        result = run_simulator(126, hive.inverse_map, shots=100)

        assert result['target_binary'] == '01111110'
        assert result['init_binary'] == '01000001'  # 65

    def test_simulation_counts(self):
        """Test that counts dict is present and correct."""
        hive = HiveKeyboard()
        result = run_simulator(126, hive.inverse_map, shots=100)

        assert 'counts' in result
        assert '01111110' in result['counts']
        assert result['counts']['01111110'] == 100

    def test_all_patterns_100_fidelity(self):
        """Test that all 256 patterns achieve 100% fidelity."""
        hive = HiveKeyboard()

        for pattern in range(256):
            result = run_simulator(pattern, hive.inverse_map, shots=100)
            assert result['fidelity'] == 100.0, f"Pattern {pattern} failed"

    def test_double_twist_parameter(self):
        """Test double_twist parameter."""
        hive = HiveKeyboard()

        result_single = run_simulator(126, hive.inverse_map, shots=100, double_twist=False)
        result_double = run_simulator(126, hive.inverse_map, shots=100, double_twist=True)

        # Both should achieve 100% on simulator
        assert result_single['fidelity'] == 100.0
        assert result_double['fidelity'] == 100.0


class TestRunBatch:
    """Tests for run_batch function."""

    def test_batch_simulation(self):
        """Test batch simulation."""
        hive = HiveKeyboard()
        patterns = [51, 126, 155, 247]
        results = run_batch(patterns, hive.inverse_map, backend='simulator', shots=100)

        assert len(results) == 4
        assert all(r['fidelity'] == 100.0 for r in results)
        assert [r['target'] for r in results] == patterns

    def test_batch_empty(self):
        """Test batch with empty list."""
        hive = HiveKeyboard()
        results = run_batch([], hive.inverse_map, backend='simulator')

        assert results == []

    def test_batch_single(self):
        """Test batch with single pattern."""
        hive = HiveKeyboard()
        results = run_batch([126], hive.inverse_map, backend='simulator', shots=100)

        assert len(results) == 1
        assert results[0]['target'] == 126


# Hardware tests are skipped by default (require IBM credentials)
class TestRunHardware:
    """Tests for run_hardware function (skipped without credentials)."""

    @pytest.mark.skip(reason="Requires IBM Quantum credentials")
    def test_hardware_submission(self):
        """Test hardware job submission."""
        pass
