"""
N-Qubit Scaling for Hive Keyboard.
Delegates to compiled Addressing Engine.
(c) 2026 Subvurs Research. Proprietary and Confidential.
"""

from typing import Dict

try:
    from .hive_engine import inverse_engine, get_engine_layers, get_engine_critical
    _HAS_ENGINE = True
except ImportError:
    _HAS_ENGINE = False


def calculate_inverse_n(target_index: int, n_qubits: int) -> int:
    """Calculate the inverse permutation for N-qubit Hive."""
    if n_qubits < 3:
        raise ValueError("N-qubit Hive requires at least 3 qubits")
    if not _HAS_ENGINE:
        raise RuntimeError("Compiled engine not available")
    return inverse_engine(target_index, n_qubits)


def generate_inverse_map(n_qubits: int) -> Dict[int, int]:
    """Generate complete inverse map for N-qubit Hive."""
    num_patterns = 1 << n_qubits
    return {i: calculate_inverse_n(i, n_qubits) for i in range(num_patterns)}


def verify_bijection(n_qubits: int) -> bool:
    """Verify that the N-qubit inverse map is a bijection."""
    inverse_map = generate_inverse_map(n_qubits)
    values = list(inverse_map.values())
    return len(set(values)) == len(values)


def get_hive_layers_n(n_qubits: int) -> Dict[int, Dict]:
    """Generate Hive layer topology for N qubits."""
    if n_qubits == 8 and _HAS_ENGINE:
        return get_engine_layers()

    max_p = (1 << n_qubits) - 1
    return {i: {'name': f'L{i}', 'range': (0, 0) if i == 0 else (max_p, max_p) if i == 7 else (1, max_p - 1),
                'description': f'Layer {i}'} for i in range(8)}


def get_critical_patterns_n(n_qubits: int) -> Dict[int, str]:
    """Get critical patterns for N-qubit Hive."""
    if n_qubits == 8 and _HAS_ENGINE:
        return get_engine_critical()

    max_p = (1 << n_qubits) - 1
    return {0: 'P0', max_p // 2: f'P{max_p // 2}', max_p: f'P{max_p}'}


def get_scaling_stats(n_qubits: int) -> Dict:
    """Get statistics about N-qubit Hive scaling."""
    return {
        'n_qubits': n_qubits,
        'num_patterns': 1 << n_qubits,
        'num_layers': 8,
        'inverse_complexity': 'O(N)',
    }
