"""
Problem Encoding for Hive Keyboard.

Maps classical optimization problems to Hive Patturns.
"""

import networkx as nx
import numpy as np

def encode_maxcut(graph, n_qubits=8):
    """
    Encode a MaxCut problem instance into a target Hive Patturn via Spectral Analysis.

    Uses the eigenvector corresponding to the largest eigenvalue of the
    adjacency matrix to find an approximate MaxCut partition.

    Args:
        graph: NetworkX graph
        n_qubits: Number of qubits (bits in the target pattern)

    Returns:
        int: Target Patturn ID (0 to 2^n_qubits - 1)
    """
    # 1. Compute the Adjacency Matrix (weighted)
    A = nx.to_numpy_array(graph, weight='weight')

    if A.size == 0:
        return 0

    # 2. Spectral Decomposition
    # For Max-Cut, the eigenvector of the largest eigenvalue of the
    # adjacency matrix provides a good approximation.
    eigenvalues, eigenvectors = np.linalg.eigh(A)
    best_ev = eigenvectors[:, -1]

    # 3. Project to Discrete Partition
    # Use sign of (value - mean) for robust thresholding.
    # This avoids floating point precision issues with median comparison.
    mean_val = np.mean(best_ev)
    bits = [1 if val >= mean_val else 0 for val in best_ev]

    # 4. Handle bitstring scaling
    # We map graph nodes 0..N-1 to qubits 0..N-1.
    # The resulting integer should have the first node at the MSB position.

    # If graph has fewer nodes than qubits, pad with 0s at the end
    if len(bits) < n_qubits:
        bits = bits + [0] * (n_qubits - len(bits))
    elif len(bits) > n_qubits:
        bits = bits[:n_qubits]

    # Convert bits to integer Patturn ID (MSB-first convention)
    patturn_id = 0
    for i, bit in enumerate(bits):
        if bit:
            patturn_id |= (1 << (n_qubits - 1 - i))

    # 5. Handle sign ambiguity: spectral methods can return either partition.
    # Evaluate both pattern and complement, return the better MaxCut.
    complement_id = ((1 << n_qubits) - 1) ^ patturn_id

    cut_value = _maxcut_value(graph, patturn_id, n_qubits)
    complement_cut = _maxcut_value(graph, complement_id, n_qubits)

    if complement_cut > cut_value:
        return complement_id

    return patturn_id


def _maxcut_value(graph, pattern, n_qubits):
    """Calculate the MaxCut value for a given pattern."""
    bits = [int(b) for b in format(pattern, f'0{n_qubits}b')]
    cut_value = 0
    for i, j, data in graph.edges(data=True):
        if i < len(bits) and j < len(bits) and bits[i] != bits[j]:
            cut_value += data.get('weight', 1)
    return cut_value

def create_hive_isomorphic_graph(n_nodes=8, target_pattern=126):
    """
    Create a graph instance whose MaxCut solution IS the target pattern.
    
    This is used to demonstrate O(1) extraction: we don't solve the
    graph; we simply 'address' its solution because we built it
    to match the Hive key.
    
    Args:
        n_nodes: Number of nodes (qubits)
        target_pattern: Integer pattern that represents the optimal cut partition
        
    Returns:
        nx.Graph: A graph instance
    """
    G = nx.Graph()
    G.add_nodes_from(range(n_nodes))
    
    # Convert pattern to bitstring
    # 1 = Set A, 0 = Set B
    bits = [int(b) for b in format(target_pattern, f'0{n_nodes}b')]
    
    # Create edges:
    # Connect nodes in different sets with high weight (cut these)
    # Connect nodes in same set with low/negative weight (don't cut these)
    
    for i in range(n_nodes):
        for j in range(i + 1, n_nodes):
            if bits[i] != bits[j]:
                # Different sets: Strong edge to encourage cutting
                G.add_edge(i, j, weight=1.0)
            else:
                # Same set: Weak/Negative edge to discourage cutting
                # (For simple MaxCut, just don't add edge or add weak one)
                pass
                
    return G
