"""
Command-line interface for Hive Keyboard.

Usage:
    hive-keyboard target 126
    hive-keyboard sequence 51 126 155 247
    hive-keyboard info 126
    hive-keyboard validate
"""

import argparse
import sys
import json
from .core import HiveKeyboard, HIVE_LAYERS, CRITICAL_PATTERNS


def target_command(args):
    """Target a specific pattern."""
    hive = HiveKeyboard(
        backend=args.backend,
        double_twist=not args.no_double_twist
    )

    result = hive.target(
        args.pattern,
        shots=args.shots,
        wait=not args.no_wait
    )

    if args.json:
        # Remove non-serializable items for JSON output
        output = {k: v for k, v in result.items() if k != 'counts' or args.verbose}
        print(json.dumps(output, indent=2))
    else:
        print(f"Pattern {result['target']} ({result['target_binary']})")
        print(f"  Init: {result['init']} ({result['init_binary']})")
        print(f"  Fidelity: {result['fidelity']:.2f}%")
        print(f"  Hits: {result['hits']}/{result['shots']}")
        print(f"  Backend: {result['backend']}")
        if 'job_id' in result:
            print(f"  Job ID: {result['job_id']}")


def sequence_command(args):
    """Run a sequence of patterns."""
    hive = HiveKeyboard(
        backend=args.backend,
        double_twist=not args.no_double_twist
    )

    results = hive.sequence(args.patterns, shots=args.shots)

    if args.json:
        output = [{k: v for k, v in r.items() if k != 'counts'} for r in results]
        print(json.dumps(output, indent=2))
    else:
        print("Sequence Results:")
        for r in results:
            status = "✓" if r['fidelity'] == 100.0 else "○"
            print(f"  {status} P{r['target']:3d}: {r['fidelity']:6.2f}%")

        avg = sum(r['fidelity'] for r in results) / len(results)
        print(f"\nAverage Fidelity: {avg:.2f}%")


def creation_command(args):
    """Run the Creation Sequence (P51 → P126 → P155 → P247)."""
    hive = HiveKeyboard(
        backend=args.backend,
        double_twist=not args.no_double_twist
    )

    print("Creation Sequence: Stabilize → Ground → Reflect → Extract")
    print("=" * 50)

    results = hive.creation_sequence(shots=args.shots)
    names = ["P51 (Stabilize)", "P126 (Ground)", "P155 (Reflect)", "P247 (Extract)"]

    for name, r in zip(names, results):
        status = "✓" if r['fidelity'] == 100.0 else "○"
        print(f"  {status} {name}: {r['fidelity']:.2f}%")

    all_perfect = all(r['fidelity'] == 100.0 for r in results)
    print("=" * 50)
    print(f"Status: {'COMPLETE' if all_perfect else 'PARTIAL'}")


def info_command(args):
    """Get detailed information about a pattern."""
    hive = HiveKeyboard()
    info = hive.info(args.pattern)

    if args.json:
        print(json.dumps(info, indent=2, default=str))
    else:
        print(f"Pattern {info['pattern']} ({info['binary']})")
        print(f"  Init Pattern: {info['init_pattern']} ({info['init_binary']})")
        print(f"  Layer: {info['layer']['layer']} ({info['layer']['name']})")
        print(f"  Description: {info['layer']['description']}")
        print(f"  Is Bridge: {info['is_bridge']}")
        print(f"  Is Critical: {info['is_critical']}")
        if info['layer'].get('critical_name'):
            print(f"  Critical Name: {info['layer']['critical_name']}")
        print(f"\nCircuit:")
        print(f"  Depth: {info['circuit']['depth']}")
        print(f"  Gate Count: {info['circuit']['gate_count']}")
        print(f"  Gates: {info['circuit']['gates']}")


def layer_command(args):
    """Show Hive layer topology."""
    print("Hive Keyboard Topology (7-Layer Oblate Spheroid)")
    print("=" * 60)

    for layer_num, info in HIVE_LAYERS.items():
        low, high = info['range']
        width = high - low + 1
        bar = "█" * min(width // 5 + 1, 20)
        print(f"Layer {layer_num} ({info['name']:12s}) [{low:3d}-{high:3d}] {bar}")
        print(f"         {info['description']}")

    print("=" * 60)
    print("\nCritical Patterns:")
    for pattern, name in sorted(CRITICAL_PATTERNS.items()):
        print(f"  P{pattern:3d}: {name}")


def validate_command(args):
    """Validate all 256 patterns."""
    hive = HiveKeyboard(double_twist=not args.no_double_twist)

    print("Validating all 256 patterns...")

    failures = []
    for i in range(256):
        result = hive.target(i, shots=args.shots)
        if result['fidelity'] < 100.0:
            failures.append((i, result['fidelity']))

        if args.verbose or (i + 1) % 32 == 0:
            print(f"  Progress: {i + 1}/256", end="\r")

    print(" " * 30)  # Clear progress line

    if failures:
        print(f"\n✗ FAILURES: {len(failures)} patterns")
        for p, f in failures[:10]:
            print(f"    P{p}: {f:.2f}%")
        if len(failures) > 10:
            print(f"    ... and {len(failures) - 10} more")
    else:
        print("\n✓ ALL 256 PATTERNS: 100% FIDELITY")
        print("  Protocol 6.0 validated successfully")


def main():
    parser = argparse.ArgumentParser(
        prog='hive-keyboard',
        description='Deterministic Quantum State Addressing'
    )
    parser.add_argument('--version', action='version', version='%(prog)s 0.1.0')

    subparsers = parser.add_subparsers(dest='command', help='Commands')

    # Common arguments
    common = argparse.ArgumentParser(add_help=False)
    common.add_argument('--backend', '-b', default='simulator',
                       help='Backend (simulator or IBM name)')
    common.add_argument('--shots', '-s', type=int, default=1024,
                       help='Number of shots')
    common.add_argument('--no-double-twist', action='store_true',
                       help='Disable Protocol 6.0 double twist')
    common.add_argument('--json', '-j', action='store_true',
                       help='Output as JSON')
    common.add_argument('--verbose', '-v', action='store_true',
                       help='Verbose output')

    # target command
    target_parser = subparsers.add_parser('target', parents=[common],
                                          help='Target a specific pattern')
    target_parser.add_argument('pattern', type=int, help='Pattern number (0-255)')
    target_parser.add_argument('--no-wait', action='store_true',
                              help='Do not wait for hardware job')
    target_parser.set_defaults(func=target_command)

    # sequence command
    seq_parser = subparsers.add_parser('sequence', parents=[common],
                                       help='Run a sequence of patterns')
    seq_parser.add_argument('patterns', type=int, nargs='+',
                           help='Pattern numbers')
    seq_parser.set_defaults(func=sequence_command)

    # creation command
    creation_parser = subparsers.add_parser('creation', parents=[common],
                                            help='Run Creation Sequence')
    creation_parser.set_defaults(func=creation_command)

    # info command
    info_parser = subparsers.add_parser('info', parents=[common],
                                        help='Get pattern information')
    info_parser.add_argument('pattern', type=int, help='Pattern number (0-255)')
    info_parser.set_defaults(func=info_command)

    # layer command
    layer_parser = subparsers.add_parser('layer', help='Show Hive topology')
    layer_parser.set_defaults(func=layer_command)

    # validate command
    val_parser = subparsers.add_parser('validate', parents=[common],
                                       help='Validate all 256 patterns')
    val_parser.set_defaults(func=validate_command)

    args = parser.parse_args()

    if args.command is None:
        parser.print_help()
        sys.exit(0)

    try:
        args.func(args)
    except KeyboardInterrupt:
        print("\nInterrupted")
        sys.exit(1)
    except Exception as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)


if __name__ == '__main__':
    main()
