"""
Quantum circuit builders for Hive Keyboard.
Thin wrapper for compiled Addressing Engine.
(c) 2026 Subvurs Research. Proprietary and Confidential.
"""

from qiskit import QuantumCircuit, QuantumRegister, ClassicalRegister

try:
    from .hive_engine import build_engine_circuit
    _HAS_ENGINE = True
except ImportError:
    _HAS_ENGINE = False
    build_engine_circuit = None


def build_deterministic_circuit(init_pattern: int, n_qubits: int = 8,
                                double_twist: bool = True,
                                include_measurement: bool = True):
    """
    Build the deterministic localization circuit.

    Args:
        init_pattern: Initialization pattern (from inverse map)
        n_qubits: Number of qubits
        double_twist: Apply boundary protection
        include_measurement: Whether to add measurement gates

    Returns:
        QuantumCircuit configured to produce the target pattern
    """
    if not _HAS_ENGINE:
        raise RuntimeError("Compiled engine not available. Please rebuild.")

    qr = QuantumRegister(n_qubits, 'q')
    cr = ClassicalRegister(n_qubits, 'c')
    qc = QuantumCircuit(qr, cr)

    ops = build_engine_circuit(init_pattern, n_qubits, 1 if double_twist else 0)

    for op in ops:
        if op[0] == 'x':
            qc.x(op[1])
        elif op[0] == 'ccx':
            qc.ccx(op[1], op[2], op[3])
        elif op[0] == 'rz':
            qc.rz(op[1], op[2])
        elif op[0] == 'cx':
            qc.cx(op[1], op[2])
        elif op[0] == 'cp':
            qc.cp(op[1], op[2], op[3])

    if include_measurement:
        qc.measure(qr, cr)

    return qc


def build_sequence_circuit(init_patterns: list, n_qubits: int = 8,
                           double_twist: bool = True):
    """
    Build circuits for a sequence of patterns.

    Args:
        init_patterns: List of initialization patterns
        n_qubits: Number of qubits
        double_twist: Apply boundary protection

    Returns:
        List of QuantumCircuits
    """
    return [build_deterministic_circuit(init, n_qubits, double_twist)
            for init in init_patterns]


def get_circuit_info(qc: QuantumCircuit) -> dict:
    """Get circuit statistics."""
    return {
        'depth': qc.depth(),
        'gate_count': sum(qc.count_ops().values()),
        'gates': dict(qc.count_ops()),
        'num_qubits': qc.num_qubits,
        'num_clbits': qc.num_clbits,
    }
