"""
Subvurs Mute Button - Main Interface

The unified interface for noise-resistant quantum encoding.
"""

from typing import Dict, Optional, List
from dataclasses import dataclass

from .encoder import Encoder
from .decoder import Decoder, DecodeResult

try:
    from qiskit import QuantumCircuit
    QISKIT_AVAILABLE = True
except ImportError:
    QISKIT_AVAILABLE = False


@dataclass
class MuteButtonConfig:
    """Configuration for Mute Button."""
    mode: str = "ratio"              # "ratio" or "differential"
    n_qubits: int = 7                # Set automatically based on mode
    noise_layers: int = 0            # CNOT layers for testing


class MuteButton:
    """
    Subvurs Mute Button

    Noise-resistant quantum encoding that works on today's hardware.

    Quick Start:
        from subvurs_mute_button import MuteButton

        mb = MuteButton()

        # Create encoded logical qubits
        zero_circuit = mb.encode(0)
        one_circuit = mb.encode(1)

        # After running on quantum hardware...
        result = mb.decode(counts)
        print(f"Decoded: {result.logical_value} (confidence: {result.confidence:.1%})")

    Modes:
        - "ratio": 7 qubits, information in probability ratio
        - "differential": 6 qubits, information in group difference

    The encoding survives significant quantum noise while maintaining
    distinguishability between logical 0 and logical 1.
    """

    def __init__(self, mode: str = "ratio"):
        """
        Initialize Mute Button.

        Args:
            mode: "ratio" (default, 7 qubits) or "differential" (6 qubits)
        """
        self.mode = mode
        self._encoder = Encoder(mode=mode)
        self._decoder = Decoder(mode=mode)
        self.n_qubits = self._encoder.n_qubits

    def encode(
        self,
        logical_value: int,
        noise_layers: int = 0,
        add_measurement: bool = True
    ) -> 'QuantumCircuit':
        """
        Create encoded quantum circuit for logical value.

        Args:
            logical_value: 0 or 1
            noise_layers: CNOT layers for noise testing (default 0)
            add_measurement: Whether to include measurement (default True)

        Returns:
            QuantumCircuit with encoded logical state
        """
        if logical_value not in (0, 1):
            raise ValueError("logical_value must be 0 or 1")

        if logical_value == 0:
            circuit = self._encoder.logical_zero(add_measurement=False)
        else:
            circuit = self._encoder.logical_one(add_measurement=False)

        # Add noise layers if requested
        if noise_layers > 0:
            circuit = self._encoder.add_noise_barrier(circuit, noise_layers)

        # Add measurement if requested
        if add_measurement:
            circuit.measure(range(self.n_qubits), range(self.n_qubits))

        return circuit

    def decode(self, counts: Dict[str, int]) -> DecodeResult:
        """
        Decode measurement results to logical value.

        Args:
            counts: Measurement counts from quantum execution
                    Format: {"0000000": 1234, "0000001": 567, ...}

        Returns:
            DecodeResult with:
                - logical_value: 0 or 1
                - confidence: 0.0 to 1.0
                - raw_metric: underlying measurement value
                - mode: encoding mode used
        """
        return self._decoder.decode(counts)

    def encode_zero(self, **kwargs) -> 'QuantumCircuit':
        """Convenience method: encode logical 0."""
        return self.encode(0, **kwargs)

    def encode_one(self, **kwargs) -> 'QuantumCircuit':
        """Convenience method: encode logical 1."""
        return self.encode(1, **kwargs)

    def create_test_suite(
        self,
        depths: List[int] = [0, 10, 20]
    ) -> tuple:
        """
        Create a test suite for benchmarking.

        Args:
            depths: List of CNOT noise depths to test

        Returns:
            (circuits, metadata) tuple for submission to hardware
        """
        circuits = []
        metadata = []

        for depth in depths:
            # Logical zero
            qc = self.encode(0, noise_layers=depth)
            circuits.append(qc)
            metadata.append({
                "name": f"MUTE_ZERO_d{depth}",
                "logical": 0,
                "depth": depth,
                "mode": self.mode,
            })

            # Logical one
            qc = self.encode(1, noise_layers=depth)
            circuits.append(qc)
            metadata.append({
                "name": f"MUTE_ONE_d{depth}",
                "logical": 1,
                "depth": depth,
                "mode": self.mode,
            })

        return circuits, metadata

    def analyze_benchmark(
        self,
        results: List[Dict[str, int]],
        metadata: List[dict]
    ) -> dict:
        """
        Analyze benchmark results.

        Args:
            results: List of count dictionaries from hardware
            metadata: Metadata from create_test_suite

        Returns:
            Analysis dictionary with accuracy and degradation metrics
        """
        analysis = {
            "total": len(results),
            "correct": 0,
            "by_depth": {},
        }

        for counts, meta in zip(results, metadata):
            result = self.decode(counts)
            expected = meta["logical"]
            depth = meta["depth"]

            correct = (result.logical_value == expected)

            if correct:
                analysis["correct"] += 1

            if depth not in analysis["by_depth"]:
                analysis["by_depth"][depth] = {
                    "total": 0,
                    "correct": 0,
                    "results": []
                }

            analysis["by_depth"][depth]["total"] += 1
            if correct:
                analysis["by_depth"][depth]["correct"] += 1

            analysis["by_depth"][depth]["results"].append({
                "name": meta["name"],
                "expected": expected,
                "decoded": result.logical_value,
                "confidence": result.confidence,
                "correct": correct,
            })

        analysis["accuracy"] = analysis["correct"] / analysis["total"]

        return analysis

    def __repr__(self):
        return f"MuteButton(mode='{self.mode}', n_qubits={self.n_qubits})"
