"""
Subvurs Mute Button - Encoder Module

Creates noise-resistant encoded quantum circuits.
"""

from typing import Optional

try:
    from qiskit import QuantumCircuit
    QISKIT_AVAILABLE = True
except ImportError:
    QISKIT_AVAILABLE = False

# Import protected core
try:
    from ._core import (
        _get_theta_zero,
        _get_theta_one,
        _get_encoding_params,
    )
    CORE_AVAILABLE = True
except ImportError:
    CORE_AVAILABLE = False
    # Fallback will raise error on use


def _check_dependencies():
    """Check required dependencies are available."""
    if not QISKIT_AVAILABLE:
        raise ImportError("Qiskit required: pip install qiskit")
    if not CORE_AVAILABLE:
        raise ImportError(
            "Core module not compiled. Run: python setup.py build_ext --inplace"
        )


class Encoder:
    """
    Subvurs Mute Button Encoder

    Creates quantum circuits with noise-resistant encoding.

    Two encoding modes:
    - Ratio encoding (7 qubits): Information stored in probability ratio
    - Differential encoding (6 qubits): Information stored in group difference

    Usage:
        encoder = Encoder()
        qc = encoder.logical_zero()  # Create |0⟩_L
        qc = encoder.logical_one()   # Create |1⟩_L
    """

    def __init__(self, mode: str = "ratio"):
        """
        Initialize encoder.

        Args:
            mode: "ratio" (default) or "differential"
        """
        _check_dependencies()

        if mode not in ("ratio", "differential"):
            raise ValueError("Mode must be 'ratio' or 'differential'")

        self.mode = mode
        self._params = _get_encoding_params()

        if mode == "ratio":
            self.n_qubits = self._params['n_qubits_ratio']
        else:
            self.n_qubits = self._params['n_qubits_diff']

    def logical_zero(self, add_measurement: bool = True) -> QuantumCircuit:
        """
        Create logical |0⟩_L state.

        Args:
            add_measurement: Whether to add measurement gates

        Returns:
            QuantumCircuit configured for logical zero
        """
        if self.mode == "ratio":
            return self._ratio_zero(add_measurement)
        else:
            return self._diff_zero(add_measurement)

    def logical_one(self, add_measurement: bool = True) -> QuantumCircuit:
        """
        Create logical |1⟩_L state.

        Args:
            add_measurement: Whether to add measurement gates

        Returns:
            QuantumCircuit configured for logical one
        """
        if self.mode == "ratio":
            return self._ratio_one(add_measurement)
        else:
            return self._diff_one(add_measurement)

    def _ratio_zero(self, add_measurement: bool) -> QuantumCircuit:
        """Create ratio-encoded logical zero."""
        qc = QuantumCircuit(self.n_qubits, self.n_qubits)
        theta = _get_theta_zero()

        for q in range(self.n_qubits):
            qc.ry(theta, q)

        if add_measurement:
            qc.measure(range(self.n_qubits), range(self.n_qubits))

        return qc

    def _ratio_one(self, add_measurement: bool) -> QuantumCircuit:
        """Create ratio-encoded logical one."""
        qc = QuantumCircuit(self.n_qubits, self.n_qubits)
        theta = _get_theta_one()

        for q in range(self.n_qubits):
            qc.ry(theta, q)

        if add_measurement:
            qc.measure(range(self.n_qubits), range(self.n_qubits))

        return qc

    def _diff_zero(self, add_measurement: bool) -> QuantumCircuit:
        """Create differential-encoded logical zero."""
        qc = QuantumCircuit(self.n_qubits, self.n_qubits)
        theta = _get_theta_zero()

        # Both groups at same angle
        for q in range(self.n_qubits):
            qc.ry(theta, q)

        if add_measurement:
            qc.measure(range(self.n_qubits), range(self.n_qubits))

        return qc

    def _diff_one(self, add_measurement: bool) -> QuantumCircuit:
        """Create differential-encoded logical one."""
        qc = QuantumCircuit(self.n_qubits, self.n_qubits)
        theta_zero = _get_theta_zero()
        theta_one = _get_theta_one()

        # Group A (qubits 0-2) at zero angle
        for q in range(3):
            qc.ry(theta_zero, q)

        # Group B (qubits 3-5) at one angle
        for q in range(3, 6):
            qc.ry(theta_one, q)

        if add_measurement:
            qc.measure(range(self.n_qubits), range(self.n_qubits))

        return qc

    def add_noise_barrier(
        self,
        circuit: QuantumCircuit,
        layers: int = 10
    ) -> QuantumCircuit:
        """
        Add CNOT noise layers to test encoding resilience.

        This applies CNOT-CNOT pairs (identity operation) that expose
        the circuit to real two-qubit gate noise.

        Args:
            circuit: Circuit to add noise to
            layers: Number of CNOT pair layers

        Returns:
            Circuit with noise layers added (before measurement)
        """
        # Remove measurements if present
        has_measurements = circuit.count_ops().get('measure', 0) > 0

        if has_measurements:
            # Create new circuit without measurements
            qc = QuantumCircuit(circuit.num_qubits, circuit.num_clbits)
            for inst, qargs, cargs in circuit.data:
                if inst.name != 'measure':
                    qc.append(inst, qargs, cargs)
        else:
            qc = circuit.copy()

        # Add noise layers
        if layers > 0:
            qc.barrier()
            for _ in range(layers):
                for q in range(qc.num_qubits - 1):
                    qc.cx(q, q + 1)
                    qc.cx(q, q + 1)
                qc.barrier()

        # Re-add measurements
        if has_measurements:
            qc.measure(range(qc.num_qubits), range(qc.num_clbits))

        return qc
